# -*- coding: iso-8859-1 -*-
"""
    MoinMoin - arnica_slides action

    This action is used to rotate, move to bak or to slide through the images from arnica.

    Based on gallery2image, (c) 2005-2008 by MoinMoin:ReimarBauer.

    @copyright: 2008 by MoinMoin:ReimarBauer
    @license: GNU GPL, see COPYING for details.
"""
import os

from MoinMoin import log
logging = log.getLogger(__name__)

from MoinMoin import config, wikiutil
from MoinMoin.Page import Page
from MoinMoin.action import ActionBase, AttachFile, cache
from MoinMoin.formatter.text_html import Formatter
from MoinMoin.packages import packLine, unpackLine
from MoinMoin.util import filesys, timefuncs

try:
    import Image
except ImportError:
    Image = None

action_name = __name__.split('.')[-1]


def image_rotate(file_name, angle, img_type):
    """ rotates the image and dependent images

    @param file_name: the name of the attachment
    @param angle: rotation angle
    @param img_type: type of image (PNG or JPEG)
    """
    # only image needs to be rotated, webnail and thumbnail
    # will be recreated by the arnica parser
    image_object = Image.open(file_name)
    os.remove(file_name)
    image_object.rotate(angle).save(file_name, img_type)


def html_js(request, counter):
    """ javascript for the slides

    @param request: request object
    @param counter: index of first slide image
    """
    html = """
<script type="text/javascript">
<!-- Original:  Ricocheting (ricocheting@hotmail.com) -->
<!-- Web Site:  http://www.ricocheting.com -->

<!-- This script and many more are available free online at -->
<!-- The JavaScript Source!! http://javascript.internet.com -->

<!-- Modifications by MoinMoin:ReimarBauer -->
<!-- 2005-10-29 -->
<!-- Many thanks to Ricocheting, it is much easier as my own one. I like it -->
<!-- Some code added and replaced for the MoinMoin:arnica parser-->


<!-- Begin

var rotate_delay = 5; // delay in milliseconds (5000 = 5 secs)
var current = %(counter)s;
var theImages = new Array();
var preloaded_images = new Array();
var AttachFile_action = '?action=AttachFile&do=get&target=';

function load_image(index) {
    var list = document.slideform.webnail_list.value;
    var value = list.split("|");
    if (preloaded_images[index] == false) {
        theImages[index].src = value[index];
        preloaded_images[index] = true;
    }
}

function arnica_preload() {
   var list = document.slideform.webnail_list.value;
   var value = list.split("|");
   var n = value.length;

   for (i = 0; i <  n-1; i++){
       theImages[i] = new Image();
       preloaded_images[i] = false;
   }
}

function arnica_getserver() {
     var value = document.URL;
     var text = value.split("/");
     return text[0]+'//'+text[2];
}

function arnica_add_comments() {
  var alias_text = document.slideform.alias.value;
  var exif_date_text = document.slideform.exif_date.value;
  var original_images = document.slideform.original_images.value;
  var index = document.slideform.slide.selectedIndex;
  var alias = alias_text.split("|");
  var exif = exif_date_text.split("|");
  images = original_images.split("|");

  document.getElementById("arnica_alias_text").innerHTML = '<a href=' + document.URL + AttachFile_action + images[index] + '>' + alias[index] +'</a>';
  document.getElementById("arnica_exif_date_text").innerHTML = exif[index];
}

function arnica_next_slide() {
   if (document.slideform.slide[current+1]) {
      load_image(current+1);
      document.images.show.src = theImages[current+1].src;
      document.slideform.slide.selectedIndex = ++current;
      arnica_add_comments();
   }
   else arnica_first_slide();
   document.getElementById("arnica_first_slide").innerHTML = '<a class="first_slide" onclick="arnica_first_slide();" name="fs"  title="first slide" >';
   document.getElementById("arnica_last_slide").innerHTML = '<a class="last_slide" onclick="arnica_last_slide();" name="fs"  title="last slide" >';
}

function arnica_previous_slide() {
   if (current-1 >= 0) {
      load_image(current-1);
      document.images.show.src = theImages[current-1].src;
      document.slideform.slide.selectedIndex = --current;
      arnica_add_comments();
   }
   else arnica_last_slide();
   document.getElementById("arnica_first_slide").innerHTML = '<a class="first_slide" onclick="arnica_first_slide();" name="fs"  title="first slide" >';
   document.getElementById("arnica_last_slide").innerHTML = '<a class="last_slide" onclick="arnica_last_slide();" name="fs"  title="last slide" >';
}

function arnica_first_slide() {
   current = 0;
   load_image(current);
   document.images.show.src = theImages[0].src;
   document.slideform.slide.selectedIndex = 0;
   arnica_add_comments();
   document.getElementById("arnica_first_slide").innerHTML = '<a class="first_slide_disabled" title="first slide" >';
   document.getElementById("arnica_last_slide").innerHTML = '<a class="last_slide" onclick="arnica_last_slide();" name="fs"  title="last slide" >';
}

function arnica_last_slide() {
   current = document.slideform.slide.length-1;
   load_image(current);
   document.images.show.src = theImages[current].src;
   document.slideform.slide.selectedIndex = current;
   arnica_add_comments();
   document.getElementById("arnica_first_slide").innerHTML = '<a class="first_slide" onclick="arnica_first_slide();" name="fs"  title="first slide" >';
   document.getElementById("arnica_last_slide").innerHTML = '<a class="last_slide_disabled" title="last slide" >';
}

function arnica_ap(text) {
   document.slideform.slidebutton.value = (text == "Stop") ? "Start" : "Stop";
   arnica_rotate();
}

function arnica_change() {
   current = document.slideform.slide.selectedIndex;
   load_image(current);
   document.images.show.src = theImages[current].src;
   arnica_add_comments();
   document.getElementById("arnica_first_slide").innerHTML = '<a class="first_slide" onclick="arnica_first_slide();" name="fs"  title="first slide" >';
   document.getElementById("arnica_last_slide").innerHTML = '<a class="last_slide" onclick="arnica_last_slide();" name="fs"  title="last slide" >';
}

function arnica_rotate() {
   if (document.slideform.slidebutton.value == "Stop") {
      current = (current == document.slideform.slide.length-1) ? 0 : current+1;
      load_image(current);
      document.images.show.src = theImages[current].src;
      document.slideform.slide.selectedIndex = current;
      arnica_add_comments();
      document.getElementById("arnica_first_slide").innerHTML = '<a class="first_slide" onclick="arnica_first_slide();" name="fs"  title="first slide" >';
      document.getElementById("arnica_last_slide").innerHTML = '<a class="last_slide" onclick="arnica_last_slide();" name="fs"  title="last slide" >';
      rotate_delay = document.slideform.duration.value * 1000.;
      window.setTimeout("arnica_rotate()", rotate_delay);
   }
}
//  End -->
</script>
""" % {
        'htdocs': request.cfg.url_prefix_static,
        'counter': counter,
    }
    return html


def html_slideform(request, pagename, alias, exif_date, images, original_images, idx, image_for_webnail):
    """ html code for the slideform

    @param request: request object
    @param pagename: pagename where the attachments are located
    @param alias: text alias for filename
    @param exif_date: date information
    @param images: list of webnail image IDs (either cache keys or attachment names)
    @param original_images: list of original image attachment names (thumbnails and webnails are derived from it)
    @param idx: index position of the image
    @param image_for_webnail: True: use attachments, False: use cache
    """
    if not image_for_webnail:
        image_url = cache.url(request, images[idx])
        this_webnail_list = [cache.url(request, name) for name in images]
    else:
        image_url = AttachFile.getAttachUrl(pagename, images[idx], request)
        this_webnail_list = [AttachFile.getAttachUrl(pagename, name, request) for name in images]

    this_webnail_list = "".join(["%s|" % url for url in this_webnail_list])

    link_to_image = AttachFile.getAttachUrl(pagename, original_images[idx], request)
    option_webnail = option_list(image_url, pagename, images, original_images, request, image_for_webnail)
    inner_table_style = ' style="border-style:none; margin:10px;"'

    html = """
<div class="arnica-slides">
<div class="navigation-animation">
<form name=slideform method="POST" action="">
<div>
<input type="hidden" name="flag" value="webnail">
<input type="hidden" name="webnail_list" value="%(this_webnail_list)s">
<input type="hidden" name="original_images" value="%(original_images)s">
<input type="hidden" name="webnail_name" value="%(this_webnail_name)s">
<input type="hidden" name="alias" value="%(this_alias_list)s">
<input type="hidden" name="exif_date" value="%(this_exif_date_list)s">
<label>Slide: <select name="slide" onChange="arnica_change();" >
%(option_webnails)s</select></label>
<input type="button" name="slidebutton" onClick="arnica_ap(this.value);" value="Start" title="AutoPlay">
<label>Duration (sec):<input type="text" name="duration" value="3.0" size="3"></label>
</div>
</form>
</div>
<div class="navigation-button">
<span id="arnica_first_slide"><a class="first_slide" onclick="arnica_first_slide();" name="fs"  title="first slide" ></a></span>
<a class="previous_slide" onclick="arnica_previous_slide();"  title="previous slide"></a>
<a class="next_slide" onClick="arnica_next_slide();"  title="next slide"></a>
<span id="arnica_last_slide"><a class="last_slide" onClick="arnica_last_slide();" title="last slide" ></a></span>
</div>
<div class="image">
<img src="%(server)s%(this_image)s" name="show" alt="%(this_alias_text)s,%(this_exif_date_text)s">
</div>
<div class="image-description">
<span id="arnica_alias_text"><a href=%(link_url)s>%(this_alias_text)s</a></span>,
<span id="arnica_exif_date_text">%(this_exif_date_text)s</span>
</div>
</div>
""" % {
        "server": request.getQualifiedURL(),
        "htdocs": request.cfg.url_prefix_static,
        "base_url": request.getScriptname(),
        "link_url": link_to_image,
        "this_webnail_list": this_webnail_list,
        "this_webnail_name": packLine(images),
        "original_images": packLine(original_images),
        "this_alias_text": wikiutil.escape(alias[idx], quote=1),
        "this_alias_list": wikiutil.escape(packLine(alias), quote=1),
        "this_exif_date_text": wikiutil.escape(exif_date[idx], quote=1),
        "this_exif_date_list": wikiutil.escape(packLine(exif_date), quote=1),
        "this_image": image_url,
        "pagename": pagename,
        "tablestyle": inner_table_style,
        "option_webnails": option_webnail,
    }
    return html


def option_list(this_image, pagename, filenames, original_images, request, image_for_webnail):
    """ generates the pulldown option list

    @param this_image: selected image on top
    @param pagename: pagename of attachments
    @param filenames: list of filenames
    @param original_images: list of original image attachment names (thumbnails and webnails are derived from it)
    @param request: request object
    @param image_for_webnail: True: use attachments, False: use cache
    """
    # TODO: original image needs to be sent to the action and used instead of the alias
    options = []
    i = 0
    for filename in filenames:
        if not image_for_webnail:
            url = cache.url(request, filename)
            prefix = 'Webnail:'
        else:
            url = AttachFile.getAttachUrl(pagename, filename, request)
            prefix = ''

        options.append('<option%(selected)s value="%(name)s">%(prefix)s %(alias)s' % {
            "selected": (url == this_image) and ' selected' or '',
            "name": this_image,
            "prefix": prefix,
            "alias": original_images[i],
        })
        i += 1
    return ''.join(options)


class arnica_slides(ActionBase):
    """ arnica_slides page action

    Note: the action name is the class name
    """
    def __init__(self, request, pagename):
        ActionBase.__init__(self, pagename, request)
        self.request = request
        self.pagename = pagename
        self.page = Page(request, pagename)
        self.method = 'POST'
        self.enctype = 'multipart/form-data'

    def is_excluded(self):
        """ Return True if action is excluded """
        return self.actionname in self.cfg.actions_excluded

    def is_allowed(self):
        """ Return True if read access is allowed """
        may = self.request.user.may
        return may.read(self.pagename)

    def do_action(self):
        """ Do the action and either return error msg or None, if there was no error. """
        return None

    def do_action_finish(self, success):
        """ Do the action and either return error msg or None, if there was no error. """
        return None

    def slide_show(self, pagename):
        """ run the slide show """
        _ = self.request.getText
        request = self.request
        if not request.user.may.read(pagename):
            msg = _("You are not allowed to view this page!")
            request.theme.add_msg(msg, "error")
            Page(request, pagename).send_page()
            return

        target = request.form.get('target', [''])[0]
        images = request.form.get('images', [''])[0]
        original_images = request.form.get('original_images', [''])[0]

        all_description = request.form.get('alias', [''])[0]
        all_exif_date = request.form.get('exif_date', [''])[0]
        image_for_webnail = request.form.get("image_for_webnail", [0])[0] == u'True'

        if not (target and images and all_description and all_exif_date):
            msg = _("Missing required parameters!")
            request.theme.add_msg(msg, "error")
            Page(request, pagename).send_page()
            return

        images = unpackLine(images)
        images = images[1:]

        original_images = unpackLine(original_images)
        original_images = original_images[1:]

        all_description = unpackLine(all_description)
        all_description = all_description[1:]

        all_exif_date = unpackLine(all_exif_date)
        all_exif_date = all_exif_date[1:]

        # XXX Check that all lists have same length
        try:
            idx = images.index(target)
        except ValueError:
            msg = _("Attachment '%(filename)s' does not exist!") % {'filename': target}
            request.theme.add_msg(msg, "error")
            Page(request, pagename).send_page()
            return

        web = {}
        if not image_for_webnail:
            web['src'] = cache.url(request, target)
        else:
            web['src'] = AttachFile.getAttachUrl(pagename, target, request)

        web['title'] = target

        mimetype = "text/html"
        request.emit_http_headers(["Content-Type: %s; charset=%s" % (mimetype, config.charset)])

        request.theme.send_title(pagename,
                                 pagename=pagename,
                                 body_onload="arnica_preload();",
                                 html_head=html_js(request, idx))

        request.write(request.formatter.startContent("content"))
        html = html_slideform(request, pagename, all_description, all_exif_date, images, original_images, idx, image_for_webnail)
        request.write(html)
        request.write(request.formatter.endContent())
        request.write(request.theme.send_footer(pagename))
        request.theme.send_closing_html()

    def render(self):
        """ executes the commands of the form data """
        _ = self.request.getText
        request = self.request
        pagename = request.form.get('pagename', [self.pagename])[0]
        page = Page(request, pagename)
        msg = None

        request.formatter = Formatter(request)
        command = request.form.get('do', ['none'])[0]
        target = request.form.get('target', [''])[0]

        # View Slides
        if command == 'slide_show':
            self.slide_show(pagename)

        # only users which are allowed to delete should use this tool
        elif request.user.may.delete(pagename):
            # only POST continues
            if not request.request_method == 'POST':
                msg = _('Missing parameters or GET method used')
                request.theme.add_msg(msg, "error")
                self.page.send_page()
                return

            filename, ext = os.path.splitext(target)

            if ext in ('.gif', '.png'):
                img_type = 'PNG'
                thumbfile = "tmp.thumbnail_%(file)s.png" % {"file": filename}
                webnail = "tmp.webnail_%(file)s.png" % {"file": filename}
            else:
                img_type = "JPEG"
                thumbfile = "tmp.thumbnail_%(file)s.jpg"  % {"file": filename}
                webnail = "tmp.webnail_%(file)s.jpg"  % {"file": filename}

            if command in ('delete', 'rotate_left', 'rotate_right', ):
                cache.remove(self.request, webnail)
                cache.remove(self.request, thumbfile)

            infile = AttachFile.getFilename(request, pagename, target)

            # removes attachment (moves to bak)
            if command == 'delete':
                if os.path.exists(infile + '.bak'):
                    os.unlink("%(file)s.bak" % {"file": infile})
                filesys.rename(infile, "%(file)s.bak" % {"file": infile})
                msg = _('%(target)s deleted, backup in place') % {'target': target}

            # rotate left
            elif command == 'rotate_left':
                image_rotate(infile, 90, img_type)
                msg = _('%(target)s rotated left 90 degrees') % {'target': target}

            # rotate right
            elif command == 'rotate_right':
                image_rotate(infile, 270, img_type)
                msg = _('%(target)s rotated right 90 degrees') % {'target': target}

            else:
                # fallback
                msg = _('action not implemented: %s') % (command, )
        else:
            msg = _('Your are not allowed to change attachments on this page: %s') % (pagename, )

        if msg:
            request.theme.add_msg(msg, "info")
            self.page.send_page()


def execute(pagename, request):
    """ Main dispatcher for the arnica_slides action. """
    _ = request.getText
    if not Image:
        msg = _('The action %(action)s needs Python Imaging Library (PIL) installed') % {'action': action_name}
        request.theme.add_msg(msg, "error")
        Page(request, pagename).send_page()
        return

    return arnica_slides(request, pagename).render()

